<?php
// /api/booking_requests.php
require_once 'db_config.php';
setApiHeaders();
$pdo = getDbConnection();
$method = $_SERVER['REQUEST_METHOD'];

try {
    switch ($method) {
        case 'GET':
            $sql = "SELECT * FROM Booking_Requests";
            $params = [];
            if (isset($_GET['id'])) {
                $sql .= " WHERE id = ?";
                $params[] = $_GET['id'];
                $stmt = $pdo->prepare($sql);
                $stmt->execute($params);
                $result = $stmt->fetch();
            } else {
                if (isset($_GET['trip_id'])) { $sql .= " WHERE trip_id = ?"; $params[] = $_GET['trip_id']; }
                if (isset($_GET['passenger_id'])) { $sql .= " WHERE passenger_id = ?"; $params[] = $_GET['passenger_id']; }
                if (isset($_GET['driver_id'])) { $sql .= " WHERE driver_id = ?"; $params[] = $_GET['driver_id']; }
                $stmt = $pdo->prepare($sql);
                $stmt->execute($params);
                $result = $stmt->fetchAll();
            }
            echo json_encode($result);
            break;

        case 'POST':
            $data = getJsonInput();
            if (empty($data['trip_id']) || empty($data['passenger_id']) || empty($data['driver_id'])) {
                http_response_code(400); echo json_encode(['message' => 'trip_id, passenger_id y driver_id son requeridos.']); exit();
            }
            // Expira en 5 minutos por defecto
            $expires_at = date('Y-m-d H:i:s', time() + (5 * 60));
            $sql = "INSERT INTO Booking_Requests (trip_id, passenger_id, driver_id, seats_requested, pickup_location, message, status, expires_at) VALUES (?, ?, ?, ?, ?, ?, ?, ?)";
            $stmt = $pdo->prepare($sql);
            $stmt->execute([
                $data['trip_id'],
                $data['passenger_id'],
                $data['driver_id'],
                $data['seats_requested'] ?? 1,
                isset($data['pickup_location']) ? json_encode($data['pickup_location']) : null,
                $data['message'] ?? null,
                'pending',
                $expires_at
            ]);
            http_response_code(201);
            echo json_encode(['message' => 'Solicitud de reserva enviada.', 'id' => $pdo->lastInsertId()]);
            break;

        case 'PUT':
            if (!isset($_GET['id'])) { http_response_code(400); echo json_encode(['message' => 'ID de solicitud no especificado.']); exit(); }
            $id = $_GET['id'];
            $data = getJsonInput();

            if (empty($data['status'])) { http_response_code(400); echo json_encode(['message' => 'El campo "status" es requerido.']); exit(); }

            if ($data['status'] == 'accepted') {
                $pdo->beginTransaction();
                try {
                    // 1. Obtener datos de la solicitud y del viaje asociado (con bloqueo)
                    $sql = "SELECT br.*, t.available_seats, t.price_per_seat FROM Booking_Requests br JOIN Trips t ON br.trip_id = t.id WHERE br.id = ? AND br.status = 'pending' FOR UPDATE";
                    $stmt = $pdo->prepare($sql);
                    $stmt->execute([$id]);
                    $request = $stmt->fetch();

                    if (!$request) {
                        throw new Exception("Solicitud no encontrada, ya procesada o inválida.", 404);
                    }

                    // 2. Verificar si hay asientos suficientes
                    if ($request['available_seats'] < $request['seats_requested']) {
                        throw new Exception("No hay suficientes asientos disponibles en el viaje.", 409);
                    }

                    // 3. Actualizar la solicitud a "aceptada"
                    $stmt_update_req = $pdo->prepare("UPDATE Booking_Requests SET status = 'accepted', response_message = ?, responded_at = NOW() WHERE id = ?");
                    $stmt_update_req->execute([$data['response_message'] ?? 'Solicitud aceptada', $id]);

                    // 4. Calcular precios (comisión del 15% como ejemplo)
                    $total_price = $request['price_per_seat'] * $request['seats_requested'];
                    $platform_fee = $total_price * 0.15;
                    $driver_earning = $total_price - $platform_fee;

                    // 5. Crear la reserva (Booking)
                    $stmt_create_booking = $pdo->prepare("INSERT INTO Bookings (trip_id, passenger_id, seats_booked, total_price, platform_fee, driver_earning, status, payment_method, pickup_address) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)");
                    $stmt_create_booking->execute([
                        $request['trip_id'],
                        $request['passenger_id'],
                        $request['seats_requested'],
                        $total_price,
                        $platform_fee,
                        $driver_earning,
                        'confirmed',
                        'cash', // Asumimos un método por defecto
                        json_decode($request['pickup_location'], true)['address'] ?? null
                    ]);
                    
                    // 6. Actualizar los asientos disponibles en el viaje
                    $stmt_update_trip = $pdo->prepare("UPDATE Trips SET available_seats = available_seats - ? WHERE id = ?");
                    $stmt_update_trip->execute([$request['seats_requested'], $request['trip_id']]);
                    
                    $pdo->commit();
                    echo json_encode(['message' => 'Solicitud aceptada y reserva creada exitosamente.']);
                    
                } catch (Exception $e) {
                    $pdo->rollBack();
                    throw $e;
                }
            } else {
                // Manejar otros estados como 'rejected', 'cancelled', 'expired'
                $sql = "UPDATE Booking_Requests SET status = ?, response_message = ?, responded_at = NOW() WHERE id = ?";
                $stmt = $pdo->prepare($sql);
                $stmt->execute([$data['status'], $data['response_message'] ?? null, $id]);
                echo json_encode(['message' => 'Estado de la solicitud actualizado.']);
            }
            break;
            
        case 'DELETE':
            http_response_code(405);
            echo json_encode(['message' => 'Método no permitido. Use PUT para cambiar el estado a "cancelado" o "rechazado".']);
            break;

        default:
            http_response_code(405);
            echo json_encode(['message' => 'Método no permitido.']);
            break;
    }
} catch (Exception $e) {
    if ($pdo->inTransaction()) {
        $pdo->rollBack();
    }
    $code = is_int($e->getCode()) && $e->getCode() > 0 ? $e->getCode() : 500;
    http_response_code($code);
    echo json_encode(['message' => $e->getMessage()]);
}
?>