<?php
// /api/bookings.php
require_once 'db_config.php';
setApiHeaders();
$pdo = getDbConnection();
$method = $_SERVER['REQUEST_METHOD'];

try {
    switch ($method) {
        case 'GET':
            $sql = "SELECT *, ST_AsText(pickup_coordinates) as pickup_coordinates FROM Bookings";
            $params = [];
            if (isset($_GET['id'])) {
                $sql .= " WHERE id = ?";
                $params[] = $_GET['id'];
                $stmt = $pdo->prepare($sql);
                $stmt->execute($params);
                $result = $stmt->fetch();
                if ($result) {
                    echo json_encode($result);
                } else {
                    http_response_code(404);
                    echo json_encode(['message' => 'Reserva no encontrada.']);
                }
            } elseif (isset($_GET['trip_id'])) {
                $sql .= " WHERE trip_id = ?";
                $params[] = $_GET['trip_id'];
                $stmt = $pdo->prepare($sql);
                $stmt->execute($params);
                echo json_encode($stmt->fetchAll());
            } elseif (isset($_GET['passenger_id'])) {
                $sql .= " WHERE passenger_id = ?";
                $params[] = $_GET['passenger_id'];
                $stmt = $pdo->prepare($sql);
                $stmt->execute($params);
                echo json_encode($stmt->fetchAll());
            } else {
                $stmt = $pdo->query($sql);
                echo json_encode($stmt->fetchAll());
            }
            break;

        case 'POST':
            $data = getJsonInput();
            
            // Validación de datos de entrada
            if (empty($data['trip_id']) || empty($data['passenger_id']) || empty($data['seats_booked']) || empty($data['total_price']) || !isset($data['platform_fee']) || !isset($data['driver_earning']) || empty($data['payment_method'])) {
                http_response_code(400);
                echo json_encode(['message' => 'Faltan campos requeridos para crear la reserva.']);
                exit();
            }

            $pdo->beginTransaction();

            try {
                // 1. Verificar si hay asientos disponibles en el viaje (con bloqueo para concurrencia)
                $stmt_check = $pdo->prepare("SELECT available_seats FROM Trips WHERE id = ? FOR UPDATE");
                $stmt_check->execute([$data['trip_id']]);
                $trip = $stmt_check->fetch();

                if (!$trip || $trip['available_seats'] < $data['seats_booked']) {
                    throw new Exception('No hay suficientes asientos disponibles o el viaje no existe.', 409); // 409 Conflict
                }
                
                // 2. Insertar la nueva reserva
                $pickup_point = isset($data['pickup_coordinates']) ? 'POINT(' . $data['pickup_coordinates']['lng'] . ' ' . $data['pickup_coordinates']['lat'] . ')' : null;
                $sql = "INSERT INTO Bookings (trip_id, passenger_id, seats_booked, total_price, platform_fee, driver_earning, status, payment_method, payment_proof_url, pickup_address, pickup_coordinates) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ST_PointFromText(?))";
                $stmt = $pdo->prepare($sql);
                $stmt->execute([
                    $data['trip_id'],
                    $data['passenger_id'],
                    $data['seats_booked'],
                    $data['total_price'],
                    $data['platform_fee'],
                    $data['driver_earning'],
                    $data['status'] ?? 'pending_payment',
                    $data['payment_method'],
                    $data['payment_proof_url'] ?? null,
                    $data['pickup_address'] ?? null,
                    $pickup_point
                ]);
                $booking_id = $pdo->lastInsertId();

                // 3. Actualizar los asientos disponibles en la tabla Trips
                $stmt_update_trip = $pdo->prepare("UPDATE Trips SET available_seats = available_seats - ? WHERE id = ?");
                $stmt_update_trip->execute([$data['seats_booked'], $data['trip_id']]);

                $pdo->commit();
                
                http_response_code(201);
                echo json_encode(['message' => 'Reserva creada exitosamente.', 'id' => $booking_id]);

            } catch (Exception $e) {
                $pdo->rollBack();
                throw $e; // Re-lanzar para el catch principal
            }
            break;

        case 'PUT':
            if (!isset($_GET['id'])) { http_response_code(400); echo json_encode(['message' => 'ID de reserva no especificado.']); exit(); }
            $id = $_GET['id'];
            $data = getJsonInput();

            // Lógica para actualizar el estado de la reserva, especialmente la cancelación
            if (isset($data['status']) && in_array($data['status'], ['cancelled_by_passenger', 'cancelled_by_driver'])) {
                
                $pdo->beginTransaction();

                try {
                    // 1. Obtener la reserva para saber cuántos asientos devolver y a qué viaje
                    $stmt_get_booking = $pdo->prepare("SELECT trip_id, seats_booked, status FROM Bookings WHERE id = ? FOR UPDATE");
                    $stmt_get_booking->execute([$id]);
                    $booking = $stmt_get_booking->fetch();

                    if (!$booking) {
                        throw new Exception("Reserva no encontrada.", 404);
                    }
                    
                    // 2. Verificar que la reserva no estuviera ya cancelada o completada
                    if (in_array($booking['status'], ['cancelled_by_passenger', 'cancelled_by_driver', 'completed'])) {
                        $pdo->rollBack();
                        http_response_code(409); // Conflict
                        echo json_encode(['message' => 'La reserva ya se encontraba en un estado final (cancelada o completada) y no puede ser modificada.']);
                        exit();
                    }

                    // 3. Actualizar el estado de la reserva
                    $sql_update_booking = "UPDATE Bookings SET status = ?, cancellation_reason = ?, updated_at = NOW() WHERE id = ?";
                    $stmt_update_booking = $pdo->prepare($sql_update_booking);
                    $stmt_update_booking->execute([$data['status'], $data['cancellation_reason'] ?? 'Cancelado por el usuario', $id]);

                    // 4. Devolver los asientos al viaje correspondiente
                    $sql_update_trip = "UPDATE Trips SET available_seats = available_seats + ? WHERE id = ?";
                    $stmt_update_trip = $pdo->prepare($sql_update_trip);
                    $stmt_update_trip->execute([$booking['seats_booked'], $booking['trip_id']]);

                    // 5. Confirmar la transacción
                    $pdo->commit();
                    echo json_encode(['message' => 'Reserva cancelada exitosamente y asientos devueltos al viaje.']);

                } catch (Exception $e) {
                    $pdo->rollBack();
                    throw $e;
                }

            } else {
                // Actualización de otros campos (calificaciones, comentarios, etc.)
                $allowed_fields = ['passenger_rating', 'driver_rating', 'passenger_comment', 'driver_comment', 'payment_proof_url'];
                $set_parts = [];
                $values = [];
                foreach ($data as $key => $value) {
                    if (in_array($key, $allowed_fields)) {
                        $set_parts[] = "$key = ?";
                        $values[] = $value;
                    }
                }
                if (empty($set_parts)) { http_response_code(400); echo json_encode(['message' => 'Ningún campo válido para actualizar.']); exit(); }

                $sql = "UPDATE Bookings SET " . implode(', ', $set_parts) . ", updated_at = NOW() WHERE id = ?";
                $values[] = $id;
                $stmt = $pdo->prepare($sql);
                $stmt->execute($values);
                echo json_encode(['message' => 'Reserva actualizada.']);
            }
            break;

        case 'DELETE':
            http_response_code(405);
            echo json_encode(['message' => 'Método no permitido. Use PUT para cambiar el estado a "cancelado".']);
            break;

        default:
            http_response_code(405);
            echo json_encode(['message' => 'Método no permitido.']);
            break;
    }
} catch (Exception $e) {
    if ($pdo->inTransaction()) {
        $pdo->rollBack();
    }
    $code = is_int($e->getCode()) && $e->getCode() > 0 ? $e->getCode() : 500;
    http_response_code($code);
    echo json_encode(['message' => $e->getMessage()]);
}
?>