<?php
// /api/driver_payments.php
require_once 'db_config.php';
setApiHeaders();
$pdo = getDbConnection();
$method = $_SERVER['REQUEST_METHOD'];

try {
    switch ($method) {
        case 'GET':
            $sql = "SELECT * FROM Driver_Payments";
            $params = [];
            if (isset($_GET['id'])) {
                $sql .= " WHERE id = ?";
                $params[] = $_GET['id'];
            } elseif (isset($_GET['driver_id'])) {
                $sql .= " WHERE driver_id = ?";
                $params[] = $_GET['driver_id'];
            }
            $stmt = $pdo->prepare($sql);
            $stmt->execute($params);
            $result = isset($_GET['id']) ? $stmt->fetch() : $stmt->fetchAll();
            echo json_encode($result);
            break;

        case 'POST':
            $data = getJsonInput();
            if (empty($data['driver_id']) || empty($data['amount'])) {
                http_response_code(400);
                echo json_encode(['message' => 'driver_id y amount son requeridos.']);
                exit();
            }
            $sql = "INSERT INTO Driver_Payments (driver_id, amount, payment_method, reference_number, payment_proof, status) VALUES (?, ?, ?, ?, ?, ?)";
            $stmt = $pdo->prepare($sql);
            $stmt->execute([
                $data['driver_id'],
                $data['amount'],
                $data['payment_method'] ?? 'transfer',
                $data['reference_number'] ?? null,
                $data['payment_proof'] ?? null,
                'pending'
            ]);
            http_response_code(201);
            echo json_encode(['message' => 'Registro de pago enviado para verificación.', 'id' => $pdo->lastInsertId()]);
            break;

        case 'PUT':
            if (!isset($_GET['id'])) { http_response_code(400); echo json_encode(['message' => 'ID de pago no especificado.']); exit(); }
            $id = $_GET['id'];
            $data = getJsonInput();

            if (empty($data['status'])) { http_response_code(400); echo json_encode(['message' => 'El campo "status" es requerido.']); exit(); }

            if ($data['status'] == 'verified') {
                if(empty($data['verified_by'])) { http_response_code(400); echo json_encode(['message' => 'El campo "verified_by" (ID del admin) es requerido para la verificación.']); exit(); }

                $pdo->beginTransaction();
                try {
                    // 1. Obtener el pago para saber el monto y el conductor (con bloqueo)
                    $stmt_get_payment = $pdo->prepare("SELECT * FROM Driver_Payments WHERE id = ? AND status = 'pending' FOR UPDATE");
                    $stmt_get_payment->execute([$id]);
                    $payment = $stmt_get_payment->fetch();

                    if (!$payment) {
                        throw new Exception("Pago no encontrado o ya ha sido procesado.", 404);
                    }

                    // 2. Actualizar el estado del pago
                    $sql_update_payment = "UPDATE Driver_Payments SET status = 'verified', admin_notes = ?, verified_at = NOW(), verified_by = ? WHERE id = ?";
                    $stmt_update_payment = $pdo->prepare($sql_update_payment);
                    $stmt_update_payment->execute([$data['admin_notes'] ?? 'Verificado por el administrador', $data['verified_by'], $id]);

                    // 3. Reducir la deuda del conductor en su perfil.
                    // Usamos GREATEST(0, ...) para evitar que la deuda se vuelva negativa por error.
                    $sql_update_debt = "UPDATE Driver_Profiles SET total_debt = GREATEST(0, total_debt - ?) WHERE user_id = ?";
                    $stmt_update_debt = $pdo->prepare($sql_update_debt);
                    $stmt_update_debt->execute([$payment['amount'], $payment['driver_id']]);

                    $pdo->commit();
                    echo json_encode(['message' => 'Pago verificado y deuda del conductor actualizada.']);

                } catch (Exception $e) {
                    $pdo->rollBack();
                    throw $e;
                }
            } else {
                // Para manejar el rechazo u otros estados que no afectan la deuda
                $sql = "UPDATE Driver_Payments SET status = ?, admin_notes = ? WHERE id = ?";
                $stmt = $pdo->prepare($sql);
                $stmt->execute([$data['status'], $data['admin_notes'] ?? null, $id]);
                echo json_encode(['message' => 'Estado del pago actualizado.']);
            }
            break;

        case 'DELETE':
            http_response_code(405);
            echo json_encode(['message' => 'Método no permitido. Los registros de pago no deben ser eliminados.']);
            break;

        default:
            http_response_code(405);
            echo json_encode(['message' => 'Método no permitido.']);
            break;
    }
} catch (Exception $e) {
    if ($pdo->inTransaction()) {
        $pdo->rollBack();
    }
    $code = is_int($e->getCode()) && $e->getCode() > 0 ? $e->getCode() : 500;
    http_response_code($code);
    echo json_encode(['message' => $e->getMessage()]);
}
?>