<?php
// /api/trip_manager.php
require_once 'db_config.php';
setApiHeaders();
$pdo = getDbConnection();

$action = $_POST['action'] ?? '';

// --- LÓGICA PRINCIPAL ---
try {
    switch ($action) {
        case 'search_for_driver':
            searchForDriver($pdo);
            break;
        case 'request_driver':
            requestDriver($pdo);
            break;
        case 'get_status':
            getStatus($pdo);
            break;
        case 'accept_trip':
            acceptTrip($pdo);
            break;
        case 'driver_arrived':
            updateBookingStatus($pdo, 'driver_arrived');
            break;
        case 'passenger_paid':
            updateBookingStatus($pdo, 'awaiting_confirmation');
            break;
        case 'confirm_payment':
            confirmPaymentAndCompleteTrip($pdo);
            break;
        case 'rate_trip':
            rateTrip($pdo);
            break;
        case 'toggle_online_status':
            toggleOnlineStatus($pdo);
            break;
        case 'update_driver_location':
            updateDriverLocation($pdo);
            break;
        default:
            http_response_code(400);
            echo json_encode(['error' => 'Acción no válida.']);
    }
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => $e->getMessage()]);
}


// --- FUNCIONES DE LÓGICA ---

function searchForDriver($pdo) {
    $lat = $_POST['lat'];
    $lng = $_POST['lng'];

    // Buscar cualquier conductor disponible (sin importar distancia)
    // Solo verificamos que esté activo y tenga deuda menor a 50,000
    $sql = "
        SELECT 
            u.id, u.full_name, u.phone_number, dp.avg_rating, dp.last_location_lat, dp.last_location_lng,
            v.brand, v.model, v.plate_number
        FROM Users u
        JOIN Driver_Profiles dp ON u.id = dp.user_id
        JOIN Vehicles v ON u.id = v.driver_id AND v.is_active = 1
        WHERE 
            u.user_type = 'driver' 
            AND dp.is_active = 1 
            AND dp.total_debt <= 50000
            AND dp.last_location_lat IS NOT NULL
        ORDER BY dp.avg_rating DESC, u.id ASC
        LIMIT 1
    ";
    $stmt = $pdo->prepare($sql);
    $stmt->execute();
    $driver = $stmt->fetch(PDO::FETCH_ASSOC);

    if ($driver) {
        // Precio de ejemplo: 5000 COP
        $driver['price'] = 5000;
        echo json_encode($driver);
    } else {
        http_response_code(404);
        echo json_encode(['error' => 'No hay conductores disponibles en este momento.']);
    }
}

function requestDriver($pdo) {
    $passengerId = $_POST['passenger_id'];
    $driverId = $_POST['driver_id'];
    $price = $_POST['price'];

    // Comisión del 25%
    $platform_fee = $price * 0.25;
    $driver_earning = $price - $platform_fee;
    
    $pickup_coords = json_decode($_POST['pickup_coords'], true);
    $pickup_point = 'POINT(' . $pickup_coords['lng'] . ' ' . $pickup_coords['lat'] . ')';
    $destination_point = 'POINT(' . $pickup_coords['lng'] . ' ' . $pickup_coords['lat'] . ')'; // Temporal, se actualizará

    $pdo->beginTransaction();
    try {
        // 1. Crear el Trip primero
        $sql_trip = "INSERT INTO Trips 
                        (driver_id, vehicle_id, direction, departure_time, total_seats, available_seats, price_per_seat, 
                         origin_address, destination_address, origin_coordinates, destination_coordinates, status) 
                     VALUES 
                        (?, (SELECT id FROM Vehicles WHERE driver_id = ? AND is_active = 1 LIMIT 1), 
                         'on_demand', NOW(), 1, 1, ?, ?, ?, ST_PointFromText(?), ST_PointFromText(?), 'scheduled')";
        
        $stmt_trip = $pdo->prepare($sql_trip);
        $stmt_trip->execute([
            $driverId, $driverId, $price, 
            $_POST['pickup_address'], $_POST['destination_address'] ?? 'Destino por definir',
            $pickup_point, $destination_point
        ]);
        $trip_id = $pdo->lastInsertId();

        // 2. Crear el Booking con el trip_id válido
        $sql_booking = "INSERT INTO Bookings 
                            (trip_id, passenger_id, seats_booked, total_price, platform_fee, driver_earning, status, payment_method, pickup_address, pickup_coordinates) 
                         VALUES 
                            (?, ?, 1, ?, ?, ?, 'pending_request', 'cash', ?, ST_PointFromText(?))";
        
        $stmt_booking = $pdo->prepare($sql_booking);
        $stmt_booking->execute([$trip_id, $passengerId, $price, $platform_fee, $driver_earning, $_POST['pickup_address'], $pickup_point]);
        
        $pdo->commit();
        echo json_encode(['success' => true, 'booking_id' => $pdo->lastInsertId(), 'trip_id' => $trip_id]);
    } catch (Exception $e) {
        $pdo->rollBack();
        throw $e;
    }
}

function acceptTrip($pdo) {
    $bookingId = $_POST['booking_id'];
    $driverId = $_POST['driver_id'];

    // El conductor acepta, la reserva pasa a estado 'confirmed' (viaje en curso)
    $sql = "UPDATE Bookings b 
            JOIN Trips t ON b.trip_id = t.id 
            SET b.status = 'confirmed' 
            WHERE b.id = ? AND t.driver_id = ?";
    $stmt = $pdo->prepare($sql);
    $stmt->execute([$bookingId, $driverId]);

    // Desactivamos al conductor para que no reciba más viajes mientras está en uno.
    $stmt_deactivate = $pdo->prepare("UPDATE Driver_Profiles SET is_active = 0 WHERE user_id = ?");
    $stmt_deactivate->execute([$driverId]);
    
    echo json_encode(['success' => true]);
}

function updateBookingStatus($pdo, $newStatus) {
    $bookingId = $_POST['booking_id'];
    $sql = "UPDATE Bookings SET status = ? WHERE id = ?";
    $stmt = $pdo->prepare($sql);
    $stmt->execute([$newStatus, $bookingId]);
    echo json_encode(['success' => true]);
}

function confirmPaymentAndCompleteTrip($pdo) {
    $bookingId = $_POST['booking_id'];

    $pdo->beginTransaction();
    try {
        // 1. Obtener datos de la reserva para la comisión
        $sql_get_booking = "SELECT b.platform_fee, t.driver_id 
                           FROM Bookings b 
                           JOIN Trips t ON b.trip_id = t.id 
                           WHERE b.id = ?";
        $stmt_get_booking = $pdo->prepare($sql_get_booking);
        $stmt_get_booking->execute([$bookingId]);
        $booking = $stmt_get_booking->fetch(PDO::FETCH_ASSOC);

        if (!$booking) throw new Exception("Booking no encontrada");

        $platform_fee = $booking['platform_fee'];
        $driver_id = $booking['driver_id'];

        // 2. Marcar reserva como completada
        $stmt_complete = $pdo->prepare("UPDATE Bookings SET status = 'completed', completion_date = NOW() WHERE id = ?");
        $stmt_complete->execute([$bookingId]);
        
        // 3. Añadir la comisión a la deuda del conductor
        $stmt_debt = $pdo->prepare("UPDATE Driver_Profiles SET total_debt = total_debt + ? WHERE user_id = ?");
        $stmt_debt->execute([$platform_fee, $driver_id]);
        
        $pdo->commit();
        echo json_encode(['success' => true]);
    } catch (Exception $e) {
        $pdo->rollBack();
        throw $e;
    }
}


function rateTrip($pdo) {
    $bookingId = $_POST['booking_id'];
    $rating = $_POST['rating'];
    $rated_by_role = $_POST['role']; // 'student' o 'driver'
    
    $pdo->beginTransaction();
    try {
        // 1. Obtener IDs de la reserva
        $stmt_get = $pdo->prepare("SELECT passenger_id, trip_id as driver_id FROM Bookings WHERE id = ?");
        $stmt_get->execute([$bookingId]);
        $ids = $stmt_get->fetch(PDO::FETCH_ASSOC);

        $target_user_id = null;
        $rating_column = '';

        if ($rated_by_role === 'student') {
            $target_user_id = $ids['driver_id'];
            $rating_column = 'driver_rating';
        } else {
            $target_user_id = $ids['passenger_id'];
            $rating_column = 'passenger_rating';
        }
        
        // 2. Actualizar la calificación en la reserva
        $stmt_rate = $pdo->prepare("UPDATE Bookings SET $rating_column = ? WHERE id = ?");
        $stmt_rate->execute([$rating, $bookingId]);

        // 3. Si el calificado es un conductor, actualizar su perfil
        if ($rated_by_role === 'student') {
            $stmt_get_profile = $pdo->prepare("SELECT avg_rating, total_ratings FROM Driver_Profiles WHERE user_id = ? FOR UPDATE");
            $stmt_get_profile->execute([$target_user_id]);
            $profile = $stmt_get_profile->fetch(PDO::FETCH_ASSOC);

            // Calcular nuevo promedio
            $new_total_ratings = $profile['total_ratings'] + 1;
            $new_avg_rating = (($profile['avg_rating'] * $profile['total_ratings']) + $rating) / $new_total_ratings;

            $stmt_update_profile = $pdo->prepare("UPDATE Driver_Profiles SET avg_rating = ?, total_ratings = ? WHERE user_id = ?");
            $stmt_update_profile->execute([$new_avg_rating, $new_total_ratings, $target_user_id]);
        }
        
        // El conductor puede volver a estar activo
        $stmt_activate = $pdo->prepare("UPDATE Driver_Profiles SET is_active = 0 WHERE user_id = ?"); // Ponemos 0 para que él decida conectarse
        $stmt_activate->execute([$ids['driver_id']]);

        $pdo->commit();
        echo json_encode(['success' => true]);
    } catch (Exception $e) {
        $pdo->rollBack();
        throw $e;
    }
}


function getStatus($pdo) {
    $role = $_POST['role'];
    $id = $_POST['id'];
    
    $sql = "";
    if ($role === 'student') {
        // Estudiante busca una reserva activa donde es el pasajero
        $sql = "SELECT b.*, ST_AsText(b.pickup_coordinates) as pickup_coords_text, 
                       u.full_name as driver_name, u.phone_number as driver_phone, 
                       dp.last_location_lat as driver_lat, dp.last_location_lng as driver_lng
                FROM Bookings b 
                JOIN Trips t ON b.trip_id = t.id
                JOIN Users u ON t.driver_id = u.id
                JOIN Driver_Profiles dp ON u.id = dp.user_id
                WHERE b.passenger_id = ? AND b.status NOT IN ('completed', 'cancelled_by_driver', 'cancelled_by_passenger')
                ORDER BY b.id DESC LIMIT 1";
    } else { // driver
        // Conductor busca una reserva activa que él debe atender
        $sql = "SELECT b.*, ST_AsText(b.pickup_coordinates) as pickup_coords_text, 
                       u.full_name as passenger_name, u.phone_number as passenger_phone
                FROM Bookings b
                JOIN Trips t ON b.trip_id = t.id
                JOIN Users u ON b.passenger_id = u.id
                WHERE t.driver_id = ? AND b.status NOT IN ('completed', 'cancelled_by_driver', 'cancelled_by_passenger')
                ORDER BY b.id DESC LIMIT 1";
    }
    
    $stmt = $pdo->prepare($sql);
    $stmt->execute([$id]);
    $trip = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if ($trip) {
        echo json_encode($trip);
    } else {
        echo json_encode(null); // No hay viaje activo
    }
}


function toggleOnlineStatus($pdo) {
    $driverId = $_POST['driver_id'];
    $newStatus = $_POST['status']; // 'online' o 'offline'
    $lat = $_POST['lat'];
    $lng = $_POST['lng'];

    if ($newStatus === 'online') {
        // Verificar si tiene deudas antes de conectarse
        $stmt_check = $pdo->prepare("SELECT total_debt FROM Driver_Profiles WHERE user_id = ?");
        $stmt_check->execute([$driverId]);
        $profile = $stmt_check->fetch(PDO::FETCH_ASSOC);

        if ($profile && $profile['total_debt'] > 50000) {
            http_response_code(403); // Forbidden
            echo json_encode(['error' => 'No puedes conectarte. Tu deuda excede los 50,000 COP.']);
            return;
        }
    }

    $is_active = ($newStatus === 'online') ? 1 : 0;
    $sql = "UPDATE Driver_Profiles SET is_active = ?, last_location_lat = ?, last_location_lng = ? WHERE user_id = ?";
    $stmt = $pdo->prepare($sql);
    $stmt->execute([$is_active, $lat, $lng, $driverId]);
    
    echo json_encode(['success' => true, 'new_status' => $is_active]);
}

function updateDriverLocation($pdo) {
    $driverId = $_POST['driver_id'];
    $lat = $_POST['lat'];
    $lng = $_POST['lng'];

    $sql = "UPDATE Driver_Profiles SET last_location_lat = ?, last_location_lng = ?, last_update = NOW() WHERE user_id = ?";
    $stmt = $pdo->prepare($sql);
    $stmt->execute([$lat, $lng, $driverId]);
    
    echo json_encode(['success' => true]);
}
?>